/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;
using System.Runtime.Serialization;

using Borland.Eco.UmlRt;
using Borland.Eco.Subscription;
using Borland.Eco.Exceptions;
using Borland.Eco.Interfaces;

namespace Borland.Eco.ObjectRepresentation
{
#region exceptions
	[Serializable]
	public class EcoObjectDeleteException: EcoException
	{
		public EcoObjectDeleteException(): base() {}
		public EcoObjectDeleteException(string message): base(message) {}
		public EcoObjectDeleteException(string message, Exception inner): base(message, inner) {}
#if !CF
		protected EcoObjectDeleteException(SerializationInfo info, StreamingContext context): base(info, context) {}
#endif
	}

	[Serializable]
	public class ArgumentNonConformException: ArgumentException
	{
		private static string FormatArgument(string fromTypeName, string toTypeName)
		{
			return InterfacesStringRes.sDoesNotConform(fromTypeName, toTypeName);
		}
		public ArgumentNonConformException(): base() {}
		public ArgumentNonConformException(string fromTypeName, string toTypeName): base(FormatArgument(fromTypeName, toTypeName)) {}
#if !CF
		protected ArgumentNonConformException(SerializationInfo info, StreamingContext context): base(info, context) {}
#endif
	}

	[Serializable]
	public class ArgumentTypeException: ArgumentException
	{
		private static string FormatArgument(Type fromType, Type toType)
		{
			if (fromType == null)
				throw new ArgumentNullException("fromType");
			if (toType == null)
				throw new ArgumentNullException("toType");
			return InterfacesStringRes.sIsNotA(fromType.FullName, toType.FullName);
		}

		private static string FormatArgument(Type toType)
		{
			if (toType == null)
				throw new ArgumentNullException("toType");
			return InterfacesStringRes.sArgumentIsNotA(toType.FullName);
		}

#if !CF
		protected ArgumentTypeException(SerializationInfo info, StreamingContext context): base(info, context) {}
#endif
		public ArgumentTypeException(): base() {}
		public ArgumentTypeException(Type fromType, Type toType): base(FormatArgument(fromType, toType)) {}
		public ArgumentTypeException(Type toType): base(FormatArgument(toType)) {}
	}

	///<summary>
	/// This Exception is raised if a trigger is called on an object with a state-machine
	/// and there is more than one transition from the current state, with that trigger and  a guard
	/// that evaluates as true.
	///</summary>
	[Serializable]
	public class AmbiguousTransitionException: EcoException
	{
		public AmbiguousTransitionException(): base() {}
		public AmbiguousTransitionException(string message): base(message) {}
		public AmbiguousTransitionException(string message, Exception inner): base(message, inner) {}
#if !CF
		protected AmbiguousTransitionException(SerializationInfo info, StreamingContext context): base(info, context) {}
#endif
	}
	///<summary>
	/// This Exception is raised if a trigger is called on an object with a state-machine
	/// and there is no transition for the trigger from the current state, with that trigger and  a guard
	/// that evaluates as true.
	///</summary>
	[Serializable]
	public class NoTransitionException: EcoException
	{
		public NoTransitionException(): base() {}
		public NoTransitionException(string message): base(message) {}
		public NoTransitionException(string message, Exception inner): base(message, inner) {}
#if !CF
		protected NoTransitionException(SerializationInfo info, StreamingContext context): base(info, context) {}
#endif
	}	
#endregion

	public abstract class ElementEqualHash
	{
		private sealed class HashCodeProviderImpl: IHashCodeProvider
		{
			int IHashCodeProvider.GetHashCode(object obj)
			{
				if (obj == null)
					throw new ArgumentNullException("obj"); // do not localize
				return ((IElement)obj).GetIsEqualHashCode();
			}
		}

		private sealed class ComparerImpl: IComparer
		{
			int IComparer.Compare(object x, object y)
			{
				IElement ex = x as IElement;
				IElement ey = y as IElement;
				if ((x == null) || (y == null))
					throw new ArgumentException(InterfacesStringRes.sCannotCompareNullValues);
				return ex.IsEqual(ey) ? 0 : 1;
			}
		}

		private readonly static IComparer m_Comparer = new ComparerImpl();
		private readonly static IHashCodeProvider m_HashCodePrivider = new HashCodeProviderImpl();

		public static IComparer Comparer
		{
			get { return m_Comparer; }
		}

		public static IHashCodeProvider HashCodeProvider
		{
			get { return m_HashCodePrivider; }
		}
	}

	///<summary>
	/// This interface is used for representing a variable in an OCL expression.
	/// In normally only occur as an element in a <see cref="Borland.Eco.ObjectRepresentation.IExternalVariableList"/>.
	///</summary>
	public interface IExternalVariable
	{
		///<summary>
		/// The static type of varaible.The value of <see cref="Borland.Eco.ObjectRepresentation.IExternalVariable.Element"/> must be euter
		/// of this type or one of its subtpyes.
		///</summary>
		IClassifier StaticUmlType { get; }
		///<summary>
		/// Subscribe to the variable. An event will be sent if the <see cref="Borland.Eco.ObjectRepresentation.IExternalVariable.Element"/>
		/// represents another IElement.
		///</summary>
		void Subscribe(ISubscriber subscriber);
		///<summary>
		/// An <see cref="Borland.Eco.ObjectRepresentation.IElement"/> which is the value of the variable.
		///</summary>
		IElement Element { get; }
		///<summary>
		/// The name of the variable.
		///</summary>
		string Name { get; }
	}

	///<summary>
	/// This interface is used for representing a collection of
	/// variables of type <see cref="Borland.Eco.ObjectRepresentation.IExternalVariable"/>, intended to be
	/// passed the mathods on the evaluator interfaces <see cref="Borland.Eco.Services.IOclService"/>,
	/// <see cref="Borland.Eco.Services.IOclPsService"/> and  <see cref="Borland.Eco.Services.IActionLanguageService"/>,
	/// The interface is returned by several of the methods on the  <see cref="Borland.Eco.Services.IVariableFactoryService"/> interface.
	///	Is is also implemented by the Borland.Eco.Handles.OclVariables>
	/// component.
	///
	///</summary>
	public interface IExternalVariableList
	{
		///<summary>
		/// Returns the number of variables in the list.
		///</summary>
		int Count { get; }

		///<summary>
		/// Returns a specified variable by index.
		///</summary>
		IExternalVariable this[int index] { get; }

		///<summary>
		/// Returns a specified variable by name.
		///</summary>
		IExternalVariable this[string name] { get; }

		///<summary>
		/// Subscribes to changes in the list of the variables (changed names, changed types, added/removed variables.
		///</summary>
		void Subscribe(ISubscriber subscriber);

		///<summary>
		/// Returns an IEnumerator for the IExternalVariable elements in the list
		///</summary>
		IEnumerator GetEnumerator();
	}
	///<summary>
	/// This is a subinterface of <see cref="Borland.Eco.ObjectRepresentation.IExternalVariableList"/>.
	/// is used for representing a collection of variables that can be modfiied by adding or removing variables.
	///</summary>
	public interface IModifiableVariableList: IExternalVariableList
	{
		///<summary>
		/// Add a new variable with name name, and value element.
		///</summary>
		void Add(string name, IElement element);
		///<summary>
		/// Create a mutable IElement of type IClassifer, and add a new variable
		/// to the collection with the name name, and the newly created IElement as a value.
		///</summary>
		void Add(string name, IClassifier umlType);
		///<summary>
		/// Create a mutable IElement of type suitable for representing values of type t, and add a new variable
		/// to the collection with the name name, and the newly created IElement as a value.
		///</summary>
		void Add(string name, Type t);
		///<summary>
		/// Create a immutable IElement of type suitable for representing value, and add a new variable
		/// to the collection with the name name, and the newly created IElement as a value.
		///</summary>
		void AddConstant(string name, object value);
		///<summary>
		/// Revove variable with name name from the collecttion.
		///</summary>
		void Remove(string name);
		///<summary>
		/// Revove variable with index i from the collection.
		///</summary>
		void RemoveAt(int i);
	}

	public enum ContentType
	{
		Primitive,
		Collection,
		Tuple,
		Object,
		ObjectList,
		ConstraintCheck
	}

	public interface IEcoServiceProvider
	{
		object GetEcoService(Type serviceType);
	}

	public interface IElementProvider
	{
		IElement Element { get; }
	}

	public interface ISubscribableElementProvider: IElementProvider
	{
		void SubscribeToElement(ISubscriber subscriber);
	}

	public interface IElement
	{
		IElement CloneValue(bool readOnly);
		bool IsEqual(IElement element);
		int GetIsEqualHashCode();
		void Assign(IElement element);
		void SubscribeToValue(ISubscriber subscriber);

		ContentType ContentType { get; }
		IElementCollection GetAsCollection();
		bool Mutable { get; }
		bool ReadOnly { get; }
		bool IsProperty { get; }
		IClassifier UmlType { get; }
		System.Object AsObject { get; set; }
	}

	public interface IFeatured: IElement
	{
		IPropertyCollection Properties { get; }
	}

	public interface IObjectOperation: IElement
	{

	}

	public interface IProperty: IElement
	{
		IStructuralFeature StructuralFeature { get; }
		IFeatured Owner { get; }
	}

	public interface IConstraintCheck: IElement
	{
		IConstraint Constraint { get; }
		IObjectInstance Subject { get; }
	}

	public interface IPrimitive: IElement
	{
		IDataType UmlDataType { get; }
	}

	public interface IVariable: IElement
	{
		IClassifier StaticUmlType { get; }
	}

	public interface IObjectContainer: IElement
	{
		void Clear();
		int Count { get; }
		IClass UmlClass { get; }
		IObjectList GetAsList();
	}

	public interface IPropertyCollection
	{
		int Count { get; }
		IProperty this[int index] { get; }
		IProperty this[UmlRt.IStructuralFeature feature] { get; }
		IProperty this[string name] { get; }
		IProperty GetByLoopbackIndex(int loopbackIndex);
		void SubscribeToPropertyChanged(ISubscriber subscriber);
	}
	
	public interface INonDeletableReason
	{
		IObject Obj { get; }
		IAssociationEnd AssociationEnd { get; }
	}

	public interface IObject: IObjectContainer, IFeatured
	{
		IEcoServiceProvider ServiceProvider { get; }
		bool Deleted { get; }
		/// <summary>
		/// IsDeletable returns true if it will be possible to delete the object. It will check 
		/// if the object has any non-empty associations with the DeleteAction "Prohibit". It will also 
		/// check recursively along any association with the DeleteAction "Cascade".
		/// </summary>
		bool IsDeletable { get; }
		/// <summary>
		/// If IsDeletable returns true, then this Property will return an object that contains a reference 
		/// to the object and the associationEnd that prevents the delete
		/// </summary>
		INonDeletableReason NonDeletableReason { get; }
		void Delete();
		IObjectInstance ObjectInstance { get; }
	}

	public interface IObjectInstance: IObject
	{
		IObjectStateMachine StateMachine { get; }
		object Invoke(IMethod method, object[] parameters);
		object Invoke(string methodName, object[] parameters);
		object Invoke(IMethod method, IElement[] parameters);
	}

	public interface IObjectReference: IObject
	{
		new IObjectInstance ObjectInstance { get; set;}
	}

	[Flags]
	public enum StateFilter
	{
		Simple= 1,
		SingleComposite = 2,
		Orthogonal= 4,
		Recursive = 8,
		AllTopLevel = Simple | SingleComposite | Orthogonal,
		All = Simple | SingleComposite | Orthogonal | Recursive
	};
	///<summary>
	/// This interface is used for manipulating the statemachine associated with
	/// an Eco object.
	///</summary>
	public interface IObjectStateMachine
	{

		///<summary>
		/// The IObjectInstance the statemachine belongs to.
		///</summary>
		IObjectInstance Owner { get;}

		///<summary>
		/// The model informmation for the statemachine
		///</summary>
		IStateMachine StateMachine { get;}
		///<summary>
		/// Execute the trigger trigger, with parameters parameters.
		/// The types of the parameters must match the declaration exactly,
		/// e.g. it is not possible to send an integer value if the formal parameter
		/// is a float.
		///</summary>
		void Trigger(ITrigger trigger, object[] parameters);
		void Trigger(ITrigger trigger, IElement[] parameters);
		void Trigger(string triggerName, object[] parameters);
		void Trigger(string triggerName, IElement[] parameters);

		bool CanTrigger(ITrigger trigger, object[] parameters);
		bool CanTrigger(ITrigger trigger, IElement[] parameters);
		bool CanTrigger(ITrigger trigger, IElement[] parameters, ISubscriber reevaluateSubscriber, ISubscriber resubscribeSubscriber);
		bool CanTrigger(string triggerName, object[] parameters);
		bool CanTrigger(string triggerName, IElement[] parameters);

		IStateCollection GetStates(StateFilter filter);
		IStateCollection GetStates(IRegion region, StateFilter filter);
		IStateCollection GetStates(string regionName, StateFilter filter);

		ITriggerCollection GetTriggers();
		ITriggerCollection GetTriggers(IRegion region);
		ITriggerCollection GetTriggers(string regionName);

		ITransitionCollection GetTransitions();
		ITransitionCollection GetTransitions(IRegion region);
		ITransitionCollection GetTransitions(string regionName);

		bool IsInState(IState state);
		bool IsInState(string stateName);

		bool IsInRegion(IRegion region);
		bool IsInRegion(string regionName);

		IRegion RegionByName(string regionName);
		IState StateByName(string stateName);
		void SubscribeToState(ISubscriber subscriber);
	}

	public interface ITuple: IFeatured
	{
		ITupleType UmlTupleType { get; }
	}

	public interface IObjectList: IObjectContainer, IEnumerable
	{
		///<exception cref="ArgumentNullException">Thrown if <paramref name="value"/> is null.</exception>
		void Add(IObject value);
		bool Contains(IObject value);
		bool SupportsAddNew { get; }
		bool SupportsRemoveAt { get; }
		IElement AddNew();
		///<exception cref="ArgumentNullException">Thrown if <paramref name="value"/> is null.</exception>
		int IndexOf(IObject value);
		///<exception cref="ArgumentNullException">Thrown if <paramref name="value"/> is null.</exception>
		void Insert(int index, IObject value);
		void Remove(IObject value);
		void RemoveAt(int index);
		void EnsureRelatedObjects(string memberName);
		///<exception cref="ArgumentNullException">Thrown if <paramref name="value"/> is null.</exception>
		IObjectInstance this[int index] { get; set; }
		bool RemoveDeletedObjects { get; set; }
	}

	public interface IQualifiedAccess
	{
		IObjectInstance GetByQualifierAndSubscribe(IElement[] Qualifiers, SubscriberPair Subscribers);
	}

	public interface IElementCollection: IElement, ICollection
	{
		void Add(IElement value);
		void Clear();
		bool Contains(IElement value);
		bool SupportsAddNew { get; }
		bool SupportsRemoveAt { get; }
		IElement AddNew();
		int IndexOf(IElement value);
		void Insert(int index, IElement value);
		void Remove(IElement value);
		void RemoveAt(int index);
		void EnsureRange(int fromIndex, int toIndex);
		IElement this[int index] { get; set; }
		ICollectionType UmlCollectionType { get; }
	}
	
	public class ElementChangedEventArgs: EventArgs
	{
		private IElement element;
		public IElement Element
		{
			get { return element; }
		}

		public ElementChangedEventArgs(IElement element)
		{
			this.element = element;
		}
	}

	public interface IObjectProvider
	{
		IObject AsIObject();
	}

	public sealed class ElementCollectionEnumerator: IEnumerator
	{
		private IElementCollection m_ElementCollection;
		private int m_Index = -1;

		public ElementCollectionEnumerator(IElementCollection elementCollection)
		{
			m_ElementCollection = elementCollection;
		}

		///<exception cref="InvalidOperationException">Thrown if there is no current.</exception>
		object IEnumerator.Current
		{
			get
			{
				if ((m_Index < 0) || ( m_Index >= m_ElementCollection.Count))
					throw new InvalidOperationException();
				return m_ElementCollection[m_Index];
			}
		}

		void IEnumerator.Reset()
		{
			m_Index = -1;
		}

		bool IEnumerator.MoveNext()
		{
			m_Index++;
			return m_Index < m_ElementCollection.Count;
		}

	}
}
